/*++

Copyright (c) 1997-2011 Microsoft Corporation

Module Name:

USBVIEW.C

Abstract:

This is the GUI goop for the USBVIEW application.

Environment:

user mode

Revision History:

04-25-97 : created
11-20-02 : minor changes to support more reporting options
04/13/2005 : major bug fixing
07/01/2008 : add UVC 1.1 support and move to Dev branch

--*/

/*****************************************************************************
I N C L U D E S
*****************************************************************************/

#include "../resource.h"
#include "../pch.h"
#include "uvcview.h"
#include "h264.h"
//#include "xmlhelper.h"

#include <commdlg.h>


/*****************************************************************************
D E F I N E S
*****************************************************************************/

// window control defines
//
#define SIZEBAR             0
#define WINDOWSCALEFACTOR   15

/*****************************************************************************
L O C A L  T Y P E D E F S
*****************************************************************************/
typedef struct _TREEITEMINFO
{
    struct _TREEITEMINFO *Next;
    USHORT Depth;
    PCHAR Name;

} TREEITEMINFO, *PTREEITEMINFO;


/*****************************************************************************
L O C A L    E N U M S
*****************************************************************************/

typedef enum _USBVIEW_SAVE_FILE_TYPE
{
    UsbViewNone = 0,
    UsbViewXmlFile,
    UsbViewTxtFile
} USBVIEW_SAVE_FILE_TYPE;

/*****************************************************************************
L O C A L    F U N C T I O N    P R O T O T Y P E S
*****************************************************************************/

//int WINAPI
//WinMain(
//    _In_ HINSTANCE hInstance,
//    _In_opt_ HINSTANCE hPrevInstance,
//    _In_ LPSTR lpszCmdLine,
//    _In_ int nCmdShow
//);

//BOOL
//CreateMainWindow(
//    int nCmdShow
//);
//
//VOID
//ResizeWindows(
//    BOOL    bSizeBar,
//    int     BarLocation
//);
//
//LRESULT CALLBACK
//MainDlgProc(
//    HWND   hwnd,
//    UINT   uMsg,
//    WPARAM wParam,
//    LPARAM lParam
//);
//
//BOOL
//USBView_OnInitDialog(
//    HWND    hWnd,
//    HWND    hWndFocus,
//    LPARAM  lParam
//);

//VOID
//USBView_OnClose(
//    HWND hWnd
//);
//
//VOID
//USBView_OnCommand(
//    HWND hWnd,
//    int  id,
//    HWND hwndCtl,
//    UINT codeNotify
//);
//
//VOID
//USBView_OnLButtonDown(
//    HWND hWnd,
//    BOOL fDoubleClick,
//    int  x,
//    int  y,
//    UINT keyFlags
//);
//
//VOID
//USBView_OnLButtonUp(
//    HWND hWnd,
//    int  x,
//    int  y,
//    UINT keyFlags
//);

//VOID
//USBView_OnMouseMove(
//    HWND hWnd,
//    int  x,
//    int  y,
//    UINT keyFlags
//);
//
//VOID
//USBView_OnSize(
//    HWND hWnd,
//    UINT state,
//    int  cx,
//    int  cy
//);
//
//LRESULT
//USBView_OnNotify(
//    HWND    hWnd,
//    int     DlgItem,
//    LPNMHDR lpNMHdr
//);
//
//BOOL
//USBView_OnDeviceChange(
//    HWND  hwnd,
//    UINT  uEvent,
//    DWORD dwEventData
//);

VOID DestroyTree(VOID);

VOID RefreshTree(VOID);

//LRESULT CALLBACK
//AboutDlgProc(
//    HWND   hwnd,
//    UINT   uMsg,
//    WPARAM wParam,
//    LPARAM lParam
//);

VOID
WalkTree(
    _In_ HTREEITEM        hTreeItem,
    _In_ LPFNTREECALLBACK lpfnTreeCallback,
    _In_opt_ PVOID            pContext
);

VOID
ExpandItem(
    HWND      hTreeWnd,
    HTREEITEM hTreeItem,
    PVOID     pContext
);

VOID
AddItemInformationToFile(
    HWND hTreeWnd,
    HTREEITEM hTreeItem,
    PVOID pContext
);

DWORD
DisplayLastError(
    _Inout_updates_bytes_(count) char    *szString,
    int     count);

//VOID AddItemInformationToXmlView(
//    HWND hTreeWnd,
//    HTREEITEM hTreeItem,
//    PVOID pContext
//);
HRESULT InitializeConsole();
VOID UnInitializeConsole();
BOOL IsStdOutFile();
VOID DisplayMessage(DWORD dwMsgId, ...);
VOID PrintString(LPTSTR lpszString);
LPTSTR WStringToAnsiString(LPWSTR lpwszString);
//VOID WaitForKeyPress();
//BOOL ProcessCommandLine();
//HRESULT ProcessCommandSaveFile(LPTSTR szFileName, DWORD dwCreationDisposition, USBVIEW_SAVE_FILE_TYPE fileType);
//HRESULT SaveAllInformationAsText(LPTSTR lpstrTextFileName, DWORD dwCreationDisposition);
//HRESULT SaveAllInformationAsXml(LPTSTR lpstrTextFileName, DWORD dwCreationDisposition);

/*****************************************************************************
G L O B A L S
*****************************************************************************/

// Flags used in dispvid.c to display default Frame descriptor for MJPEG,
//  Uncompressed, Vendor and FrameBased Formats
UCHAR   g_chMJPEGFrameDefault = 0;
UCHAR   g_chUNCFrameDefault = 0;
UCHAR   g_chVendorFrameDefault = 0;
UCHAR   g_chFrameBasedFrameDefault = 0;

// Spec version of UVC device
UINT g_chUVCversion = 0;

// Save the current Configuration starting and ending addresses
// Used in ValidateDescAddress()
//
PUSB_CONFIGURATION_DESCRIPTOR   g_pConfigDesc = 0;
PSTRING_DESCRIPTOR_NODE         g_pStringDescs = 0;
PUCHAR                          g_descEnd = 0;

int gDeviceSpeed = 0; 
DWORD dwConfigLength = 0;
DWORD dwConfigIndex = 0;
BOOL gDoConfigDesc = TRUE;
BOOL gDoAnnotation = TRUE;
BOOL gLogDebug = FALSE;
int  TotalHubs = 0;

// Base address of the USBDEVICEINFO for device we're parsing
PUSBDEVICEINFO CurrentUSBDeviceInfo = 0;

// Base address of the Configuration descriptor we're parsing
PUSB_CONFIGURATION_DESCRIPTOR  CurrentConfigDesc = 0;


extern DEVICE_GUID_LIST gHubList;
extern DEVICE_GUID_LIST gDeviceList;

/*****************************************************************************
G L O B A L S    P R I V A T E    T O    T H I S    F I L E
*****************************************************************************/

HINSTANCE       ghInstance = NULL;
HWND            ghMainWnd = NULL;
HWND            ghTreeWnd = NULL;
HWND            ghEditWnd = NULL;
HWND            ghStatusWnd = NULL;
HMENU           ghMainMenu = NULL;
HTREEITEM       ghTreeRoot = NULL;
HCURSOR         ghSplitCursor = NULL;
HDEVNOTIFY      gNotifyDevHandle = NULL;
HDEVNOTIFY      gNotifyHubHandle = NULL;
HANDLE          ghStdOut = NULL;

BOOL            gbConsoleFile = FALSE;
BOOL            gbConsoleInitialized = FALSE;
BOOL            gbButtonDown = FALSE;
BOOL            gDoAutoRefresh = TRUE;

int             gBarLocation = 0;
int             giGoodDevice = 0;
int             giBadDevice = 0;
int             giComputer = 0;
int             giHub = 0;
int             giNoDevice = 0;
int             giGoodSsDevice = 0;
int             giNoSsDevice = 0;






HRESULT InitializeConsole()
{
    HRESULT hr = S_OK;

    SetLastError(0);

    // Find if STD_OUTPUT is a console or has been redirected to a File
    gbConsoleFile = IsStdOutFile();

    if (!gbConsoleFile)
    {
        // Output is not redirected and GUI application do not have console by default, create a console
        if (AllocConsole())
        {
#pragma warning(disable:4996) // We don' need the FILE * returned by freopen
            // Reopen STDOUT , STDIN and STDERR
            if ((freopen("conout$", "w", stdout) != NULL) &&
                (freopen("conin$", "r", stdin) != NULL) &&
                (freopen("conout$", "w", stderr) != NULL))
            {
                gbConsoleInitialized = TRUE;
                ghStdOut = GetStdHandle(STD_OUTPUT_HANDLE);
            }
#pragma warning(default:4996)
        }
    }

    if (INVALID_HANDLE_VALUE == ghStdOut || FALSE == gbConsoleInitialized)
    {
        hr = HRESULT_FROM_WIN32(GetLastError());
        OOPS();
    }
    return hr;
}

/*****************************************************************************

UnInitializeConsole()

UnInitializes the console

*****************************************************************************/
VOID UnInitializeConsole()
{
    gbConsoleInitialized = FALSE;
    FreeConsole();
}

/*****************************************************************************

IsStdOutFile()

Finds if the STD_OUTPUT has been redirected to a file
*****************************************************************************/
BOOL IsStdOutFile()
{
    unsigned htype;
    HANDLE hFile;

    // 1 = STDOUT
    hFile = (HANDLE)_get_osfhandle(1);
    htype = GetFileType(hFile);
    htype &= ~FILE_TYPE_REMOTE;


    // Check if file type is character file
    if (FILE_TYPE_DISK == htype)
    {
        return TRUE;
    }

    return FALSE;
}


/*****************************************************************************

DisplayMessage()

Displays a message to standard output
*****************************************************************************/
VOID DisplayMessage(DWORD dwResId, ...)
{
    CHAR szFormat[4096];
    HRESULT hr = S_OK;
    LPTSTR lpszMessage = NULL;
    DWORD dwLen = 0;
    va_list ap;

    va_start(ap, dwResId);

    // Initialize console if needed
    if (!gbConsoleInitialized)
    {
        hr = InitializeConsole();
        if (FAILED(hr))
        {
            OOPS();
            return;
        }
    }

    // Load the string resource
    dwLen = LoadString(GetModuleHandle(NULL),
        dwResId,
        szFormat,
        ARRAYSIZE(szFormat)
    );

    if (0 == dwLen)
    {
        PrintString("Unable to find message for given resource ID");

        // Return if resource ID could not be found
        return;
    }

    dwLen = FormatMessage(
        FORMAT_MESSAGE_FROM_STRING | FORMAT_MESSAGE_ALLOCATE_BUFFER,
        szFormat,
        dwResId,
        0,
        (LPTSTR)&lpszMessage,
        ARRAYSIZE(szFormat),
        &ap);

    if (dwLen > 0)
    {
        PrintString(lpszMessage);
        LocalFree(lpszMessage);
    }
    else
    {
        PrintString("Unable to find message for given ID");
    }

    va_end(ap);
    return;
}

/*****************************************************************************

WStringToAnsiString()

Converts the Wide char string to ANSI string and returns the allocated ANSI string.
*****************************************************************************/
LPTSTR WStringToAnsiString(LPWSTR lpwszString)
{
    int strLen = 0;
    LPTSTR szAnsiBuffer = NULL;

    szAnsiBuffer =(LPTSTR) LocalAlloc(LPTR, (MAX_PATH + 1) * sizeof(CHAR));

    // Convert string from from WCHAR to ANSI
    if (NULL != szAnsiBuffer)
    {
        strLen = WideCharToMultiByte(
            CP_ACP,
            0,
            lpwszString,
            -1,
            szAnsiBuffer,
            MAX_PATH + 1,
            NULL,
            NULL);

        if (strLen > 0)
        {
            return szAnsiBuffer;
        }
    }
    return NULL;
}

/*****************************************************************************

PrintString()

Displays a string to standard output
*****************************************************************************/
VOID PrintString(LPTSTR lpszString)
{
    DWORD dwBytesWritten = 0;
    size_t Len = 0;
    LPSTR lpOemString = NULL;

    if (INVALID_HANDLE_VALUE == ghStdOut || NULL == lpszString)
    {
        OOPS();
        // Return if invalid inputs
        return;
    }

    if (FAILED(StringCchLength(lpszString, OUTPUT_MESSAGE_MAX_LENGTH, &Len)))
    {
        OOPS();
        // Return if string is too long
        return;
    }

    if (gbConsoleFile)
    {
        // Console has been redirected to a file, ex: `usbview /savexml:xx > test.txt`. We need to use WriteFile instead of
        // WriteConsole for text output.
        lpOemString = (LPSTR)LocalAlloc(LPTR, (Len + 1) * sizeof(CHAR));
        if (lpOemString != NULL)
        {
            if (CharToOemBuff(lpszString, lpOemString, (DWORD)Len))
            {
                WriteFile(ghStdOut, (LPVOID)lpOemString, (DWORD)Len, &dwBytesWritten, NULL);
            }
            else
            {
                OOPS();
            }
        }
    }
    else
    {
        // Write to std out in console
        WriteConsole(ghStdOut, (LPVOID)lpszString, (DWORD)Len, &dwBytesWritten, NULL);
    }

    return;
}

/*****************************************************************************

WaitForKeyPress()

Waits for key press in case of console
*****************************************************************************/
//VOID WaitForKeyPress()
//{
//    // Wait for key press if console
//    if (!gbConsoleFile && gbConsoleInitialized)
//    {
//        DisplayMessage(IDS_USBVIEW_PRESSKEY);
//        (VOID)_getch();
//    }
//    return;
//}

/*****************************************************************************

CreateMainWindow()

*****************************************************************************/




/*****************************************************************************

ResizeWindows()

Handles resizing the two child windows of the main window.  If
bSizeBar is true, then the sizing is happening because the user is
moving the bar.  If bSizeBar is false, the sizing is happening
because of the WM_SIZE or something like that.

*****************************************************************************/

VOID
ResizeWindows(
    BOOL    bSizeBar,
    int     BarLocation
)
{
    RECT    MainClientRect;
    RECT    MainWindowRect;
    RECT    TreeWindowRect;
    RECT    StatusWindowRect;
    int     right;

    // Is the user moving the bar?
    //
    if (!bSizeBar)
    {
        BarLocation = gBarLocation;
    }

    GetClientRect(ghMainWnd, &MainClientRect);

    GetWindowRect(ghStatusWnd, &StatusWindowRect);

    // Make sure the bar is in a OK location
    //
    if (bSizeBar)
    {
        if (BarLocation <
            GetSystemMetrics(SM_CXSCREEN) / WINDOWSCALEFACTOR)
        {
            return;
        }

        if ((MainClientRect.right - BarLocation) <
            GetSystemMetrics(SM_CXSCREEN) / WINDOWSCALEFACTOR)
        {
            return;
        }
    }

    // Save the bar location
    //
    gBarLocation = BarLocation;

    // Move the tree window
    //
    MoveWindow(ghTreeWnd,
        0,
        0,
        BarLocation,
        MainClientRect.bottom - StatusWindowRect.bottom + StatusWindowRect.top,
        TRUE);

    // Get the size of the window (in case move window failed
    //
    GetWindowRect(ghTreeWnd, &TreeWindowRect);
    GetWindowRect(ghMainWnd, &MainWindowRect);

    right = TreeWindowRect.right - MainWindowRect.left;

    // Move the edit window with respect to the tree window
    //
    MoveWindow(ghEditWnd,
        right + SIZEBAR,
        0,
        MainClientRect.right - (right + SIZEBAR),
        MainClientRect.bottom - StatusWindowRect.bottom + StatusWindowRect.top,
        TRUE);

    // Move the Status window with respect to the tree window
    //
    MoveWindow(ghStatusWnd,
        0,
        MainClientRect.bottom - StatusWindowRect.bottom + StatusWindowRect.top,
        MainClientRect.right,
        StatusWindowRect.bottom - StatusWindowRect.top,
        TRUE);
}


/*****************************************************************************

MainWndProc()

*****************************************************************************/

//LRESULT CALLBACK
//MainDlgProc(
//    HWND   hWnd,
//    UINT   uMsg,
//    WPARAM wParam,
//    LPARAM lParam
//)
//{
//
//    switch (uMsg)
//    {
//
//        HANDLE_MSG(hWnd, WM_INITDIALOG, USBView_OnInitDialog);
//        HANDLE_MSG(hWnd, WM_CLOSE, USBView_OnClose);
//        HANDLE_MSG(hWnd, WM_COMMAND, USBView_OnCommand);
//        HANDLE_MSG(hWnd, WM_LBUTTONDOWN, USBView_OnLButtonDown);
//        HANDLE_MSG(hWnd, WM_LBUTTONUP, USBView_OnLButtonUp);
//        HANDLE_MSG(hWnd, WM_MOUSEMOVE, USBView_OnMouseMove);
//        HANDLE_MSG(hWnd, WM_SIZE, USBView_OnSize);
//        HANDLE_MSG(hWnd, WM_NOTIFY, USBView_OnNotify);
//        HANDLE_MSG(hWnd, WM_DEVICECHANGE, USBView_OnDeviceChange);
//    }
//
//    return 0;
//}

/*****************************************************************************

USBView_OnInitDialog()

*****************************************************************************/


/*****************************************************************************

USBView_OnClose()

*****************************************************************************/



/*****************************************************************************

AddItemInformationToFile()

Saves the information about the current item to the list
*****************************************************************************/
VOID
AddItemInformationToFile(
    HWND hTreeWnd,
    HTREEITEM hTreeItem,
    PVOID pContext
)
{
    HRESULT hr = S_OK;
    HANDLE hf = NULL;
    DWORD dwBytesWritten = 0;

    hf = *((PHANDLE)pContext);

    ResetTextBuffer();

    hr = UpdateTreeItemDeviceInfo(hTreeWnd, hTreeItem);

    if (FAILED(hr))
    {
        OOPS();
    }
    else
    {
        WriteFile(hf, GetTextBuffer(), GetTextBufferPos() * sizeof(CHAR), &dwBytesWritten, NULL);
    }

    ResetTextBuffer();
}



/*****************************************************************************

SaveAllInformationAsText()

Saves the entire USB tree as a text file
*****************************************************************************/
HRESULT
SaveAllInformationAsText(
    LPTSTR lpstrTextFileName,
    DWORD dwCreationDisposition
)
{
    HRESULT hr = S_OK;
    HANDLE hf = NULL;

    hf = CreateFile(lpstrTextFileName,
        GENERIC_WRITE,
        0,
        NULL,
        dwCreationDisposition,
        FILE_ATTRIBUTE_NORMAL,
        NULL);

    if (hf == INVALID_HANDLE_VALUE)
    {
        hr = HRESULT_FROM_WIN32(GetLastError());
        OOPS();
    }
    else
    {
        if (GetLastError() == ERROR_ALREADY_EXISTS)
        {
            // CreateFile() sets this error if we are overwriting an existing file
            // Reset this error to avoid false alarms
            SetLastError(0);
        }

        if (ghTreeRoot == NULL)
        {
            // If tree has not been populated yet, try a refresh
            RefreshTree();
        }

        if (ghTreeRoot)
        {

            LockFile(hf, 0, 0, 0, 0);
            WalkTreeTopDown(ghTreeRoot, AddItemInformationToFile, &hf, NULL);
            UnlockFile(hf, 0, 0, 0, 0);
            CloseHandle(hf);

            hr = S_OK;
        }
        else
        {
            hr = HRESULT_FROM_WIN32(GetLastError());
            OOPS();
        }
    }

    ResetTextBuffer();
    return hr;
}


/*****************************************************************************

USBView_OnCommand()

*****************************************************************************/

/*****************************************************************************

USBView_OnLButtonDown()

*****************************************************************************/


/*****************************************************************************

USBView_OnLButtonUp()

*****************************************************************************/


/*****************************************************************************

USBView_OnMouseMove()

*****************************************************************************/


/*****************************************************************************

USBView_OnSize();

*****************************************************************************/



/*****************************************************************************

USBView_OnNotify()

*****************************************************************************/

//LRESULT
//USBView_OnNotify(
//    HWND    hWnd,
//    int     DlgItem,
//    LPNMHDR lpNMHdr
//)
//{
//    UNREFERENCED_PARAMETER(hWnd);
//    UNREFERENCED_PARAMETER(DlgItem);
//
//    if (lpNMHdr->code == TVN_SELCHANGED)
//    {
//        HTREEITEM hTreeItem;
//
//        hTreeItem = ((NM_TREEVIEW *)lpNMHdr)->itemNew.hItem;
//
//        if (hTreeItem)
//        {
//            UpdateEditControl(ghEditWnd,
//                ghTreeWnd,
//                hTreeItem);
//        }
//    }
//
//    return 0;
//}


/*****************************************************************************

USBView_OnDeviceChange()

*****************************************************************************/

//BOOL
//USBView_OnDeviceChange(
//    HWND  hwnd,
//    UINT  uEvent,
//    DWORD dwEventData
//)
//{
//    UNREFERENCED_PARAMETER(hwnd);
//    UNREFERENCED_PARAMETER(dwEventData);
//
//    if (gDoAutoRefresh)
//    {
//        switch (uEvent)
//        {
//        case DBT_DEVICEARRIVAL:
//        case DBT_DEVICEREMOVECOMPLETE:
//            RefreshTree();
//            break;
//        }
//    }
//
//    return TRUE;
//}



/*****************************************************************************

DestroyTree()

*****************************************************************************/

VOID DestroyTree(VOID)
{
    // Clear the selection of the TreeView, so that when the tree is
    // destroyed, the control won't try to constantly "shift" the
    // selection to another item.
    //
    TreeView_SelectItem(ghTreeWnd, NULL);

    // Destroy the current contents of the TreeView
    //
    if (ghTreeRoot)
    {
        WalkTree(ghTreeRoot, CleanupItem, NULL);

        TreeView_DeleteAllItems(ghTreeWnd);

        ghTreeRoot = NULL;
    }

    ClearDeviceList(&gDeviceList);
    ClearDeviceList(&gHubList);
}

/*****************************************************************************

RefreshTree()

*****************************************************************************/

VOID RefreshTree(VOID)
{
    CHAR  statusText[128];
    ULONG devicesConnected;

    // Clear the edit control
    //
    SetWindowText(ghEditWnd, "");

    // Destroy the current contents of the TreeView
    //
    DestroyTree();

    // Create the root tree node
    //
    ghTreeRoot = AddLeaf(TVI_ROOT, 0, "My Computer", ComputerIcon);

    if (ghTreeRoot != NULL)
    {
        // Enumerate all USB buses and populate the tree
        //
        EnumerateHostControllers(ghTreeRoot, &devicesConnected);

        //
        // Expand all tree nodes
        //
        WalkTree(ghTreeRoot, ExpandItem, NULL);

        // Update Status Line with number of devices connected
        //
        memset(statusText, 0, sizeof(statusText));
        StringCchPrintf(statusText, sizeof(statusText),
#ifdef H264_SUPPORT
            "UVC Spec Version: %d.%d Version: %d.%d Devices Connected: %d   Hubs Connected: %d",
            UVC_SPEC_MAJOR_VERSION, UVC_SPEC_MINOR_VERSION, USBVIEW_MAJOR_VERSION, USBVIEW_MINOR_VERSION,
            devicesConnected, TotalHubs);
#else
            "Devices Connected: %d   Hubs Connected: %d",
            devicesConnected, TotalHubs);
#endif

        SetWindowText(ghStatusWnd, statusText);
    }
    else
    {
        OOPS();
    }

}

/*****************************************************************************

AboutDlgProc()

*****************************************************************************/

//LRESULT CALLBACK
//AboutDlgProc(
//    HWND   hwnd,
//    UINT   uMsg,
//    WPARAM wParam,
//    LPARAM lParam
//)
//{
//    UNREFERENCED_PARAMETER(lParam);
//
//    switch (uMsg)
//    {
//    case WM_INITDIALOG:
//    {
//        HRESULT hr;
//        char TextBuffer[TEXT_ITEM_LENGTH];
//        HWND hItem;
//
//        hItem = GetDlgItem(hwnd, IDC_VERSION);
//
//        if (hItem != NULL)
//        {
//            hr = StringCbPrintfA(TextBuffer,
//                sizeof(TextBuffer),
//                "USBView version: %d.%d",
//                USBVIEW_MAJOR_VERSION,
//                USBVIEW_MINOR_VERSION);
//            if (SUCCEEDED(hr))
//            {
//                SetWindowText(hItem, TextBuffer);
//            }
//        }
//
//        hItem = GetDlgItem(hwnd, IDC_UVCVERSION);
//
//        if (hItem != NULL)
//        {
//            hr = StringCbPrintfA(TextBuffer,
//                sizeof(TextBuffer),
//                "USB Video Class Spec version: %d.%d",
//                UVC_SPEC_MAJOR_VERSION,
//                UVC_SPEC_MINOR_VERSION);
//            if (SUCCEEDED(hr))
//            {
//                SetWindowText(hItem, TextBuffer);
//            }
//        }
//    }
//    break;
//    case WM_COMMAND:
//
//        switch (LOWORD(wParam))
//        {
//        case IDOK:
//        case IDCANCEL:
//
//            EndDialog(hwnd, 0);
//            break;
//        }
//        break;
//
//    }
//
//    return FALSE;
//}


/*****************************************************************************

AddLeaf()

*****************************************************************************/

//HTREEITEM
//AddLeaf(
//    HTREEITEM hTreeParent,
//    LPARAM    lParam,
//    _In_ LPTSTR    lpszText,
//    TREEICON  TreeIcon
//)
//{
//
//}


/*****************************************************************************

WalkTreeTopDown()

*****************************************************************************/

VOID
WalkTreeTopDown(
    _In_ HTREEITEM        hTreeItem,
    _In_ LPFNTREECALLBACK lpfnTreeCallback,
    _In_opt_ PVOID            pContext,
    _In_opt_ LPFNTREENOTIFYCALLBACK lpfnTreeNotifyCallback
)
{
    if (hTreeItem)
    {
        HTREEITEM hTreeChild = TreeView_GetChild(ghTreeWnd, hTreeItem);
        HTREEITEM hTreeSibling = TreeView_GetNextSibling(ghTreeWnd, hTreeItem);

        //
        // Call the lpfnCallBack on the node itself.
        //
        (*lpfnTreeCallback)(ghTreeWnd, hTreeItem, pContext);

        //
        // Recursively call WalkTree on the node's first child.
        //

        if (hTreeChild)
        {
            WalkTreeTopDown(hTreeChild,
                lpfnTreeCallback,
                pContext,
                lpfnTreeNotifyCallback);
        }

        //
        // Recursively call WalkTree on the node's first sibling.
        //
        if (hTreeSibling)
        {
            WalkTreeTopDown(hTreeSibling,
                lpfnTreeCallback,
                pContext,
                lpfnTreeNotifyCallback);
        }
        else
        {
            // If there are no more siblings, we have reached the end of
            // list of child nodes. Call notify function
            if (lpfnTreeNotifyCallback != NULL)
            {
                (*lpfnTreeNotifyCallback)(pContext);
            }
        }
    }
}

/*****************************************************************************

WalkTree()

*****************************************************************************/

VOID
WalkTree(
    _In_ HTREEITEM        hTreeItem,
    _In_ LPFNTREECALLBACK lpfnTreeCallback,
    _In_opt_ PVOID            pContext
)
{
    if (hTreeItem)
    {
        // Recursively call WalkTree on the node's first child.
        //
        WalkTree(TreeView_GetChild(ghTreeWnd, hTreeItem),
            lpfnTreeCallback,
            pContext);

        //
        // Call the lpfnCallBack on the node itself.
        //
        (*lpfnTreeCallback)(ghTreeWnd, hTreeItem, pContext);

        //
        //
        // Recursively call WalkTree on the node's first sibling.
        //
        WalkTree(TreeView_GetNextSibling(ghTreeWnd, hTreeItem),
            lpfnTreeCallback,
            pContext);
    }
}

/*****************************************************************************

ExpandItem()

*****************************************************************************/

VOID
ExpandItem(
    HWND      hTreeWnd,
    HTREEITEM hTreeItem,
    PVOID     pContext
)
{
    //
    // Make this node visible.
    //
    UNREFERENCED_PARAMETER(pContext);

    TreeView_Expand(hTreeWnd, hTreeItem, TVE_EXPAND);
}

/*****************************************************************************

SaveAllInformationAsXML()

Saves the entire USB tree as an XML file
*****************************************************************************/
//HRESULT
//SaveAllInformationAsXml(
//    LPTSTR lpstrTextFileName,
//    DWORD dwCreationDisposition
//)
//{
//    HRESULT hr = S_OK;
//
//    if (ghTreeRoot == NULL)
//    {
//        // If tree has not been populated yet, try a refresh
//        RefreshTree();
//    }
//    if (ghTreeRoot)
//    {
//        WalkTreeTopDown(ghTreeRoot, AddItemInformationToXmlView, NULL, XmlNotifyEndOfNodeList);
//
//        hr = SaveXml(lpstrTextFileName, dwCreationDisposition);
//    }
//    else
//    {
//        hr = E_FAIL;
//        OOPS();
//    }
//    ResetTextBuffer();
//    return hr;
//}

//*****************************************************************************
//
//  AddItemInformationToXmlView
//
//  hTreeItem - Handle of selected TreeView item for which information should
//  be added to the XML View
//
//*****************************************************************************
//VOID
//AddItemInformationToXmlView(
//    HWND hTreeWnd,
//    HTREEITEM hTreeItem,
//    PVOID pContext
//)
//{
//    TV_ITEM tvi;
//    PVOID   info;
//    PCHAR tviName = NULL;
//
//    UNREFERENCED_PARAMETER(pContext);
//
//#ifdef H264_SUPPORT
//    ResetErrorCounts();
//#endif
//
//    tviName = (PCHAR)ALLOC(256);
//
//    if (NULL == tviName)
//    {
//        return;
//    }
//
//    //
//    // Get the name of the TreeView item, along with the a pointer to the
//    // info we stored about the item in the item's lParam.
//    //
//
//    tvi.mask = TVIF_HANDLE | TVIF_TEXT | TVIF_PARAM;
//    tvi.hItem = hTreeItem;
//    tvi.pszText = (LPSTR)tviName;
//    tvi.cchTextMax = 256;
//
//    TreeView_GetItem(hTreeWnd,
//        &tvi);
//
//    info = (PVOID)tvi.lParam;
//
//    if (NULL != info)
//    {
//        //
//        // Add Item to XML object
//        //
//        switch (*(PUSBDEVICEINFOTYPE)info)
//        {
//        case HostControllerInfo:
//            XmlAddHostController(tviName, (PUSBHOSTCONTROLLERINFO)info);
//            break;
//
//        case RootHubInfo:
//            XmlAddRootHub(tviName, (PUSBROOTHUBINFO)info);
//            break;
//
//        case ExternalHubInfo:
//            XmlAddExternalHub(tviName, (PUSBEXTERNALHUBINFO)info);
//            break;
//
//        case DeviceInfo:
//            XmlAddUsbDevice(tviName, (PUSBDEVICEINFO)info);
//            break;
//        }
//
//    }
//    return;
//}

/*****************************************************************************

DisplayLastError()

*****************************************************************************/

DWORD
DisplayLastError(
    _Inout_updates_bytes_(count) char *szString,
    int count)
{
    LPVOID lpMsgBuf;

    // get the last error code
    DWORD dwError = GetLastError();

    // get the system message for this error code
    if (FormatMessage(
        FORMAT_MESSAGE_ALLOCATE_BUFFER |
        FORMAT_MESSAGE_FROM_SYSTEM |
        FORMAT_MESSAGE_IGNORE_INSERTS,
        NULL,
        dwError,
        MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
        (LPTSTR)&lpMsgBuf,
        0,
        NULL))
    {
        StringCchPrintf(szString, count, "Error: %s", (LPTSTR)lpMsgBuf);
    }

    // Free the local buffer
    LocalFree(lpMsgBuf);

    // return the error
    return dwError;
}

#if DBG

/*****************************************************************************

Oops()

*****************************************************************************/

VOID
Oops
(
    _In_ PCHAR  File,
    ULONG       Line
)
{
    char szBuf[1024];
    LPTSTR lpMsgBuf;
    DWORD dwGLE = GetLastError();

    memset(szBuf, 0, sizeof(szBuf));

    // get the system message for this error code
    if (FormatMessage(
        FORMAT_MESSAGE_ALLOCATE_BUFFER |
        FORMAT_MESSAGE_FROM_SYSTEM,
        NULL,
        dwGLE,
        MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
        (LPTSTR)&lpMsgBuf,
        0,
        NULL))
    {
        StringCchPrintf(szBuf, sizeof(szBuf),
            "File: %s, Line %d\r\nGetLastError 0x%x %u %s\n",
            File, Line, dwGLE, dwGLE, lpMsgBuf);
    }
    else
    {
        StringCchPrintf(szBuf, sizeof(szBuf),
            "File: %s, Line %d\r\nGetLastError 0x%x %u\r\n",
            File, Line, dwGLE, dwGLE);
    }
    OutputDebugString(szBuf);

    // Free the system allocated local buffer
    LocalFree(lpMsgBuf);

    return;
}

#endif
